<?php
// PHP Twitch Stream Tracker
// This script provides a web-based dashboard for Twitch stream information.

// Error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Load configuration
$configFile = __DIR__ . '/config.php';
if (!file_exists($configFile)) {
    die('<p style="color:red;">Configuration file not found. Please copy config.example.php to config.php and configure your credentials.</p>');
}
$config = require $configFile;

// Load functions
require_once __DIR__ . '/includes/functions.php';

// Handle AJAX requests for live metrics updates
if (isset($_GET['ajax']) && $_GET['ajax'] == '1') {
    header('Content-Type: text/html; charset=utf-8');
    
if (!isset($_GET['username']) || empty($_GET['username'])) {
        exit('No username provided');
    }
    
    $broadcasterLogin = trim($_GET['username']);
    $timezone = new DateTimeZone($config['timezone']);
    
    // Get access token
    $tokenCacheKey = get_cache_key('access_token', []);
    $accessToken = get_cached_data($tokenCacheKey, 3600, $config['cache_dir']);
    
    if (!$accessToken) {
        $accessToken = get_app_access_token($config['client_id'], $config['client_secret']);
        if ($accessToken) {
            set_cached_data($tokenCacheKey, $accessToken, 3600, $config['cache_dir']);
        }
    }
    
    if (!$accessToken) {
        exit('Authentication error');
    }
    
    // Get user info
    $userInfo = get_user_info($broadcasterLogin, $accessToken, $config['client_id']);
    if (!$userInfo) {
        exit('User not found');
    }
    
    $userId = $userInfo['id'];
    
    // Get stream info
    $streamInfo = get_stream_info($userId, $accessToken, $config['client_id']);
    
    if ($streamInfo && isset($streamInfo['viewer_count'])) {
        track_live_stream_metrics($userId, $streamInfo['viewer_count'], $config['cache_dir']);
        $liveMetrics = get_live_stream_metrics($userId, $config['cache_dir']);
        
        // Output just the metrics
        ?>
        <div id="current-viewers">
            <div class="metric-value"><?php echo number_format($streamInfo['viewer_count']); ?></div>
        </div>
        <?php if ($liveMetrics): ?>
        <div id="avg-viewers">
            <div class="metric-value"><?php echo number_format($liveMetrics['average_viewers']); ?></div>
        </div>
        <div id="peak-viewers">
            <div class="metric-value"><?php echo number_format($liveMetrics['peak_viewers']); ?></div>
        </div>
        <?php endif; ?>
        <?php
    } else {
        clear_live_stream_metrics($userId, $config['cache_dir']);
    }
    
    exit();
}

// Check for broadcaster login from URL parameter
if (!isset($_GET['username']) || empty($_GET['username'])) {
    $pageTitle = 'Twitch Stream Tracker - Vertali Analytics';
    $includeCharts = false;
    include __DIR__ . '/includes/header.php';
    ?>
    <div class="container" style="grid-template-columns: 1fr; max-width: 800px; margin: 0 auto;">
        <div class="card" style="text-align: center; padding: 4rem 3rem;">
            <div style="font-size: 4rem; margin-bottom: 2rem; color: var(--primary);">
                <i class="fas fa-chart-line"></i>
            </div>
            <h1 style="font-size: 2.5rem; font-weight: 700; margin-bottom: 1rem; color: var(--text-primary); letter-spacing: -1px;">Twitch Stream Tracker</h1>
            <p style="font-size: 1.125rem; color: var(--text-tertiary); margin-bottom: 3rem; line-height: 1.7;">Comprehensive analytics and insights for Twitch streamers</p>
            
            <form method="GET" action="" style="display: flex; gap: var(--spacing-md); max-width: 600px; margin: 0 auto 3rem;">
                <input 
                    type="text" 
                    name="username" 
                    placeholder="Enter Twitch username..."
                    required
                    autofocus
                    style="flex: 1; padding: 1rem 1.5rem; background: var(--bg-secondary); border: 1px solid var(--border-secondary); color: var(--text-primary); font-size: 1rem; border-radius: var(--radius-lg); transition: all 0.2s ease; font-family: 'Inter', sans-serif;"
                    onfocus="this.style.borderColor='var(--primary)'; this.style.outline='none'; this.style.boxShadow='0 0 0 3px rgba(99, 102, 241, 0.1)';"
                    onblur="this.style.borderColor='var(--border-secondary)'; this.style.boxShadow='none';"
                >
                <button 
                    type="submit"
                    style="padding: 1rem 2rem; background: var(--primary); border: none; color: white; font-weight: 600; font-size: 1rem; cursor: pointer; border-radius: var(--radius-lg); transition: all 0.2s ease; display: flex; align-items: center; gap: var(--spacing-xs);"
                    onmouseover="this.style.background='var(--primary-dark)'; this.style.transform='translateY(-1px)'; this.style.boxShadow='var(--shadow-md)';"
                    onmouseout="this.style.background='var(--primary)'; this.style.transform='translateY(0)'; this.style.boxShadow='none';"
                >
                    <i class="fas fa-search"></i> Track
                </button>
            </form>
            
            <div style="margin-top: 3rem; padding-top: 2rem; border-top: 1px solid var(--border-secondary);">
                <p style="font-size: 0.875rem; color: var(--text-tertiary); margin-bottom: 1.5rem; text-transform: uppercase; letter-spacing: 0.5px; font-weight: 600;">Try These Streamers</p>
                <div style="display: flex; gap: var(--spacing-md); justify-content: center; flex-wrap: wrap;">
                    <a href="?username=vertali" style="color: var(--text-primary); text-decoration: none; padding: 0.625rem 1.25rem; background: var(--bg-secondary); border: 1px solid var(--border-secondary); border-radius: var(--radius-md); transition: all 0.2s ease; font-weight: 500; font-size: 0.875rem;" onmouseover="this.style.borderColor='var(--primary)'; this.style.background='var(--bg-tertiary)'; this.style.transform='translateY(-1px)';" onmouseout="this.style.borderColor='var(--border-secondary)'; this.style.background='var(--bg-secondary)'; this.style.transform='translateY(0)';">vertali</a>
                    <a href="?username=xqc" style="color: var(--text-primary); text-decoration: none; padding: 0.625rem 1.25rem; background: var(--bg-secondary); border: 1px solid var(--border-secondary); border-radius: var(--radius-md); transition: all 0.2s ease; font-weight: 500; font-size: 0.875rem;" onmouseover="this.style.borderColor='var(--primary)'; this.style.background='var(--bg-tertiary)'; this.style.transform='translateY(-1px)';" onmouseout="this.style.borderColor='var(--border-secondary)'; this.style.background='var(--bg-secondary)'; this.style.transform='translateY(0)';">xqc</a>
                    <a href="?username=ninja" style="color: var(--text-primary); text-decoration: none; padding: 0.625rem 1.25rem; background: var(--bg-secondary); border: 1px solid var(--border-secondary); border-radius: var(--radius-md); transition: all 0.2s ease; font-weight: 500; font-size: 0.875rem;" onmouseover="this.style.borderColor='var(--primary)'; this.style.background='var(--bg-tertiary)'; this.style.transform='translateY(-1px)';" onmouseout="this.style.borderColor='var(--border-secondary)'; this.style.background='var(--bg-secondary)'; this.style.transform='translateY(0)';">ninja</a>
            </div>
            </div>
        </div>
        </div>
    <?php
    include __DIR__ . '/includes/footer.php';
    exit();
}

$broadcasterLogin = trim($_GET['username']);
$pageTitle = 'Twitch Stream Tracker - ' . htmlspecialchars($broadcasterLogin);

// Initialize timezone
$timezone = new DateTimeZone($config['timezone']);
$todayDate = (new DateTime('now', $timezone))->format('Y-m-d');
$weekAgo = (new DateTime('-7 days', $timezone))->format('Y-m-d');
$monthAgo = (new DateTime('-30 days', $timezone))->format('Y-m-d');

// Get access token with caching
$tokenCacheKey = get_cache_key('access_token', []);
$accessToken = get_cached_data($tokenCacheKey, 3600, $config['cache_dir']); // Cache token for 1 hour

if (!$accessToken) {
    $accessToken = get_app_access_token($config['client_id'], $config['client_secret']);
    if ($accessToken) {
        set_cached_data($tokenCacheKey, $accessToken, 3600, $config['cache_dir']);
    }
}

if (!$accessToken) {
    $pageTitle = 'Error - Vertali Analytics';
    include __DIR__ . '/includes/header.php';
    ?>
    <div class="container">
        <div class="data-panel error-message" style="max-width: 600px; margin: 2rem auto;">
            <h2><i class="fas fa-exclamation-triangle"></i> Authentication Error</h2>
            <p>Failed to get Twitch API access token. Please check your CLIENT_ID and CLIENT_SECRET in the configuration file.</p>
        </div>
    </div>
    <?php
    include __DIR__ . '/includes/footer.php';
    exit();
}

// Get user info with caching
$userCacheKey = get_cache_key('user_info', ['username' => $broadcasterLogin]);
$userInfo = get_cached_data($userCacheKey, $config['cache_ttl'], $config['cache_dir']);

if (!$userInfo) {
    $userInfo = get_user_info($broadcasterLogin, $accessToken, $config['client_id']);
    if ($userInfo && !isset($userInfo['error'])) {
        set_cached_data($userCacheKey, $userInfo, $config['cache_ttl'], $config['cache_dir']);
    }
}

if (!$userInfo || isset($userInfo['error'])) {
    $pageTitle = 'User Not Found - Vertali Analytics';
    include __DIR__ . '/includes/header.php';
    ?>
    <div class="container">
        <div class="data-panel error-message" style="max-width: 600px; margin: 2rem auto;">
            <h2><i class="fas fa-user-slash"></i> User Not Found</h2>
            <p>Could not find user "<?php echo htmlspecialchars($broadcasterLogin); ?>". Please check the username and try again.</p>
            <?php if (isset($userInfo['error']) && $userInfo['error'] === 'rate_limit'): ?>
                <p style="margin-top: 1rem;"><strong>Note:</strong> Rate limit exceeded. Please try again later.</p>
            <?php endif; ?>
        </div>
    </div>
    <?php
    include __DIR__ . '/includes/footer.php';
    exit();
}

$userId = $userInfo['id'];
$displayName = $userInfo['display_name'];
$profileImageUrl = $userInfo['profile_image_url'];
$createdAt = new DateTime($userInfo['created_at']);

// Get stream info (shorter cache for live data)
$streamCacheKey = get_cache_key('stream_info', ['user_id' => $userId]);
$streamInfo = get_cached_data($streamCacheKey, 30, $config['cache_dir']); // Cache for 30 seconds for live data

if (!$streamInfo) {
    $streamInfo = get_stream_info($userId, $accessToken, $config['client_id']);
    if ($streamInfo && !isset($streamInfo['error'])) {
        set_cached_data($streamCacheKey, $streamInfo, 30, $config['cache_dir']);
    } elseif (!$streamInfo) {
        $streamInfo = null;
        // Clear live metrics if stream ended
        clear_live_stream_metrics($userId, $config['cache_dir']);
    }
}

// Track live stream metrics if stream is live
$liveMetrics = null;
if ($streamInfo && isset($streamInfo['viewer_count'])) {
    // Track current viewer count
    track_live_stream_metrics($userId, $streamInfo['viewer_count'], $config['cache_dir']);
    // Get calculated metrics
    $liveMetrics = get_live_stream_metrics($userId, $config['cache_dir']);
}

// Get follower count with caching
$followerCacheKey = get_cache_key('followers', ['user_id' => $userId]);
$rawFollowerCount = get_cached_data($followerCacheKey, $config['cache_ttl'], $config['cache_dir']);

if ($rawFollowerCount === null) {
    $rawFollowerCount = get_channel_followers($userId, $accessToken, $config['client_id']);
    if ($rawFollowerCount !== null && !is_array($rawFollowerCount)) {
        set_cached_data($followerCacheKey, $rawFollowerCount, $config['cache_ttl'], $config['cache_dir']);
    }
}

$followerCount = number_format($rawFollowerCount ?: 0);

// Get VODs with caching
$vodsCacheKey = get_cache_key('vods', ['user_id' => $userId, 'limit' => 100]);
$vods = get_cached_data($vodsCacheKey, $config['cache_ttl'], $config['cache_dir']);

if (!$vods) {
    $vods = get_all_vods($userId, $accessToken, $config['client_id'], 100, 'archive');
    if ($vods && !isset($vods['error'])) {
        set_cached_data($vodsCacheKey, $vods, $config['cache_ttl'], $config['cache_dir']);
    } else {
        $vods = [];
    }
}

// Calculate today's analytics
$todayVods = filter_vods_by_date($vods, $todayDate, $todayDate, $timezone);
$todayAnalytics = calculate_vod_analytics($todayVods, $timezone);

$totalSecondsToday = $todayAnalytics['total_duration'];
$totalStreamedTodayFormatted = seconds_to_duration($totalSecondsToday);
$averageViewersToday = $todayAnalytics['average_views'];
$peakViewersToday = $todayAnalytics['peak_views'];
$vodCountToday = $todayAnalytics['total_vods'];
$totalViewCountToday = $todayAnalytics['total_views'];

// Calculate weekly analytics
$weekVods = filter_vods_by_date($vods, $weekAgo, $todayDate, $timezone);
$weekAnalytics = calculate_vod_analytics($weekVods, $timezone);

// Calculate monthly analytics
$monthVods = filter_vods_by_date($vods, $monthAgo, $todayDate, $timezone);
$monthAnalytics = calculate_vod_analytics($monthVods, $timezone);

// Get popular games
$popularGames = array_slice($monthAnalytics['games'], 0, 5, true);

// Calculate additional advanced metrics
$dailyStats = [];
$hourlyPatterns = [];
$growthMetrics = ['views_growth_percent' => 0, 'this_week_views' => 0, 'last_week_views' => 0, 'this_week_streams' => 0, 'last_week_streams' => 0];
$streamSchedule = [];
$totalHoursStreamed = 0;
$viewsPerHour = 0;
$avgStreamDuration = 0;

if (!empty($vods)) {
    $dailyStats = calculate_daily_stats($vods, $timezone, 30);
    $hourlyPatterns = calculate_hourly_patterns($vods, $timezone);
    $growthMetrics = calculate_growth_metrics($vods, $timezone);
    $streamSchedule = get_stream_schedule($monthVods, $timezone);
}

// Calculate average stream duration and engagement metrics
if ($monthAnalytics['total_vods'] > 0 && $monthAnalytics['total_duration'] > 0) {
    $avgStreamDuration = (int)round($monthAnalytics['total_duration'] / $monthAnalytics['total_vods'] / 60); // in minutes
    $totalHoursStreamed = round($monthAnalytics['total_duration'] / 3600, 1);
    $viewsPerHour = $totalHoursStreamed > 0 ? (int)round($monthAnalytics['total_views'] / $totalHoursStreamed) : 0;
} else {
    $avgStreamDuration = 0;
    $totalHoursStreamed = 0;
    $viewsPerHour = 0;
}

// Include header
$includeCharts = true;
include __DIR__ . '/includes/header.php';
?>

    <header class="header boot-anim">
        <div class="header-top">
            <div class="logo-section">
                <a href="?username=<?php echo urlencode($broadcasterLogin); ?>" class="logo">
                    <div class="logo-icon-wrapper">
                        <div class="logo-icon">
                            <i class="fas fa-chart-line"></i>
                        </div>
                    </div>
                    <div class="logo-text-wrapper">
                        <div class="logo-text">Vertali Analytics</div>
                        <div class="logo-tagline">Stream Intelligence Platform</div>
                    </div>
                </a>
                <span class="beta-badge">BETA</span>
            </div>
            
            <div class="header-center">
                <div class="header-stats">
                    <?php if ($streamInfo): ?>
                    <div class="status-badge-container">
                        <div class="header-badge live">
                            <span class="header-badge-icon"></span>
                            <span>LIVE NOW</span>
                        </div>
                    </div>
                    <?php else: ?>
                    <div class="status-badge-container">
                        <div class="header-badge offline">
                            <span class="header-badge-icon"></span>
                            <span>OFFLINE</span>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <div class="header-actions">
                <a href="?" class="nav-link" title="Search Another Streamer">
                    <i class="fas fa-search"></i> 
                    <span>Search</span>
                </a>
                <div class="timestamp" id="last-updated-time">
                    <i class="fas fa-clock"></i>
                    <span><?php $now = new DateTime('now', $timezone); echo $now->format('M d, Y · h:i:s A T'); ?></span>
                </div>
            </div>
        </div>
        </header>

    <div class="dashboard-wrapper">
        <aside class="profile-sidebar">
            <div class="card profile-card boot-anim" style="animation-delay: 0.1s;">
                <div class="avatar-container">
                    <img src="<?php echo htmlspecialchars($profileImageUrl); ?>" alt="Streamer Avatar" class="avatar">
                    <div class="status-indicator <?php echo $streamInfo ? 'live' : ''; ?>"></div>
                </div>
                <h1 class="streamer-name"><?php echo htmlspecialchars($displayName); ?></h1>
                <p class="streamer-handle">@<?php echo htmlspecialchars($broadcasterLogin); ?></p>
                <div class="status-badge <?php echo $streamInfo ? 'live' : 'offline'; ?>">
                    <i class="fas fa-<?php echo $streamInfo ? 'circle' : 'circle'; ?>"></i>
                    <span><?php echo $streamInfo ? 'LIVE' : 'OFFLINE'; ?></span>
                </div>
            </div>

            <div class="card boot-anim" style="animation-delay: 0.2s;">
                <ul class="stats-list">
                    <li class="stat-item">
                        <span class="stat-label">Followers</span>
                        <span class="stat-value"><?php echo htmlspecialchars($followerCount); ?></span>
                    </li>
                    <li class="stat-item">
                        <span class="stat-label">Joined</span>
                        <span class="stat-value"><?php echo $createdAt->format('M Y'); ?></span>
                    </li>
                    <li class="stat-item">
                        <span class="stat-label">Streamed Today</span>
                        <span class="stat-value"><?php echo $totalStreamedTodayFormatted; ?></span>
                    </li>
                    <?php if ($weekAnalytics['total_vods'] > 0): ?>
                    <li class="stat-item">
                        <span class="stat-label">This Week</span>
                        <span class="stat-value"><?php echo seconds_to_duration($weekAnalytics['total_duration']); ?></span>
                    </li>
                    <?php endif; ?>
                    <?php if ($monthAnalytics['total_vods'] > 0): ?>
                    <li class="stat-item">
                        <span class="stat-label">Total Hours (30d)</span>
                        <span class="stat-value"><?php echo number_format($totalHoursStreamed); ?>h</span>
                    </li>
                    <?php endif; ?>
                    <?php if ($growthMetrics['views_growth_percent'] != 0): ?>
                    <li class="stat-item">
                        <span class="stat-label">Views Growth</span>
                        <span class="stat-value">
                            <span class="growth-indicator <?php echo $growthMetrics['views_growth_percent'] > 0 ? 'positive' : 'negative'; ?>">
                                <i class="fas fa-<?php echo $growthMetrics['views_growth_percent'] > 0 ? 'arrow-up' : 'arrow-down'; ?>"></i>
                                <?php echo abs($growthMetrics['views_growth_percent']); ?>%
                            </span>
                        </span>
                    </li>
                    <?php endif; ?>
                </ul>
            </div>
        </aside>

        <main class="main-content">
            <div class="tabs-container boot-anim">
                <button class="tab-button active" data-tab="overview">
                    <i class="fas fa-chart-pie tab-icon"></i>
                    <span>Overview</span>
                </button>
                <button class="tab-button" data-tab="vod">
                    <i class="fas fa-video tab-icon"></i>
                    <span>VOD Analytics</span>
                </button>
                <?php if ($streamInfo): ?>
                <button class="tab-button" data-tab="live">
                    <i class="fas fa-broadcast-tower tab-icon"></i>
                    <span>Live Stream</span>
                </button>
                <?php endif; ?>
                <button class="tab-button" data-tab="trends">
                    <i class="fas fa-chart-line tab-icon"></i>
                    <span>Trends</span>
                </button>
                <button class="tab-button" data-tab="content">
                    <i class="fas fa-list tab-icon"></i>
                    <span>Content</span>
                </button>
            </div>

            <!-- Overview Tab -->
            <div class="tab-content active" id="overview-tab">
                <div class="content-grid three-col">
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">Today's VODs</h2>
                        </div>
                        <div class="metrics-grid" style="grid-template-columns: 1fr;">
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($totalViewCountToday); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Total VOD Views
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-users"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($averageViewersToday); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg VOD Viewers
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $vodCountToday; ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VODs Today
                                </div>
                            </div>
                        </div>
                    </div>

                    <?php if ($weekAnalytics['total_vods'] > 0): ?>
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">This Week</h2>
                        </div>
                        <div class="metrics-grid" style="grid-template-columns: 1fr;">
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $weekAnalytics['total_vods']; ?></div>
                                <div class="metric-label">VODs</div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="metric-value"><?php echo seconds_to_duration($weekAnalytics['total_duration']); ?></div>
                                <div class="metric-label">Stream Time</div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($weekAnalytics['average_views']); ?></div>
                                <div class="metric-label">Avg Views</div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if ($monthAnalytics['total_vods'] > 0): ?>
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">This Month</h2>
                        </div>
                        <div class="metrics-grid" style="grid-template-columns: 1fr;">
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $monthAnalytics['total_vods']; ?></div>
                                <div class="metric-label">VODs</div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="metric-value"><?php echo seconds_to_duration($monthAnalytics['total_duration']); ?></div>
                                <div class="metric-label">Stream Time</div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($monthAnalytics['average_views']); ?></div>
                                <div class="metric-label">Avg Views</div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- VOD Analytics Tab -->
            <div class="tab-content" id="vod-tab">
                <div class="content-grid">
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">Today's VOD Analytics</h2>
                            <span class="section-badge vod">
                                <i class="fas fa-video"></i> Recorded
                            </span>
                        </div>
                        <div class="metrics-grid">
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($totalViewCountToday); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Total VOD Views
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-users"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($averageViewersToday); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg VOD Viewers
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-chart-line"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($peakViewersToday); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Peak VOD Viewers
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $vodCountToday; ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VODs Today
                                </div>
                            </div>
                        </div>
                    </div>

                    <?php if ($weekAnalytics['total_vods'] > 0 || $monthAnalytics['total_vods'] > 0): ?>
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">VOD Performance Overview</h2>
                            <span class="section-badge vod">
                                <i class="fas fa-video"></i> Recorded Content
                            </span>
                        </div>
                        <div class="metrics-grid">
                            <?php if ($weekAnalytics['total_vods'] > 0): ?>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $weekAnalytics['total_vods']; ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VODs (Week)
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="metric-value"><?php echo seconds_to_duration($weekAnalytics['total_duration']); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VOD Time (Week)
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($weekAnalytics['average_views']); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg VOD Views (Week)
                                </div>
                            </div>
                            <?php endif; ?>
                            <?php if ($monthAnalytics['total_vods'] > 0): ?>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-broadcast-tower"></i>
                                </div>
                                <div class="metric-value"><?php echo $monthAnalytics['total_vods']; ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VODs (Month)
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="metric-value"><?php echo seconds_to_duration($monthAnalytics['total_duration']); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VOD Time (Month)
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-eye"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($monthAnalytics['average_views']); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg VOD Views (Month)
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if ($monthAnalytics['total_vods'] > 0): ?>
                    <div class="card boot-anim vod-metrics">
                        <div class="section-header">
                            <h2 class="section-title">VOD Engagement Metrics</h2>
                            <span class="section-badge vod">
                                <i class="fas fa-video"></i> Recorded
                            </span>
                        </div>
                        <div class="metrics-grid">
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-tachometer-alt"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($viewsPerHour); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    VOD Views Per Hour
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-hourglass-half"></i>
                                </div>
                                <div class="metric-value"><?php echo $avgStreamDuration; ?>m</div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg VOD Duration
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-chart-bar"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($monthAnalytics['total_views'] / max($monthAnalytics['total_vods'], 1)); ?></div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Avg Views Per VOD
                                </div>
                            </div>
                            <div class="metric-card vod">
                                <div class="metric-icon vod">
                                    <i class="fas fa-clock"></i>
                                </div>
                                <div class="metric-value"><?php echo number_format($totalHoursStreamed); ?>h</div>
                                <div class="metric-label">
                                    <i class="fas fa-video metric-label-icon"></i>
                                    Total VOD Hours
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Live Stream Tab -->
            <?php if ($streamInfo): ?>
            <div class="tab-content" id="live-tab">
                <div class="card boot-anim live-metrics" id="live-stream-card">
                    <div class="section-header">
                        <h2 class="section-title">Live Stream Analytics</h2>
                        <span class="section-badge live">
                            <i class="fas fa-circle"></i> LIVE NOW
                        </span>
                    </div>
                    <div style="margin-bottom: var(--spacing-lg); padding: var(--spacing-md); background: var(--bg-secondary); border-radius: var(--radius-md); border-left: 3px solid var(--live-color);">
                        <div style="font-size: 1.125rem; font-weight: 600; color: var(--text-primary); margin-bottom: var(--spacing-xs);">
                            <?php echo htmlspecialchars($streamInfo['title']); ?>
                        </div>
                        <div style="font-size: 0.875rem; color: var(--text-tertiary);">
                            <i class="fas fa-gamepad" style="margin-right: 6px;"></i>
                            <?php echo htmlspecialchars($streamInfo['game_name']); ?>
                        </div>
                    </div>
                    <div class="metrics-grid">
                        <div class="metric-card live" id="current-viewers">
                            <div class="metric-icon live">
                                <i class="fas fa-eye"></i>
                            </div>
                            <div class="metric-value"><?php echo number_format($streamInfo['viewer_count']); ?></div>
                            <div class="metric-label">
                                <i class="fas fa-broadcast-tower metric-label-icon"></i>
                                Current Live Viewers
                            </div>
                        </div>
                        <?php if ($liveMetrics): ?>
                        <div class="metric-card live" id="avg-viewers">
                            <div class="metric-icon live">
                                <i class="fas fa-users"></i>
                            </div>
                            <div class="metric-value"><?php echo number_format($liveMetrics['average_viewers']); ?></div>
                            <div class="metric-label">
                                <i class="fas fa-broadcast-tower metric-label-icon"></i>
                                Avg Live Viewers
                            </div>
                        </div>
                        <div class="metric-card live" id="peak-viewers">
                            <div class="metric-icon live">
                                <i class="fas fa-chart-line"></i>
                            </div>
                            <div class="metric-value"><?php echo number_format($liveMetrics['peak_viewers']); ?></div>
                            <div class="metric-label">
                                <i class="fas fa-broadcast-tower metric-label-icon"></i>
                                Peak Live Viewers
                            </div>
                        </div>
                        <div class="metric-card live">
                            <div class="metric-icon live">
                                <i class="fas fa-clock"></i>
                            </div>
                            <div class="metric-value"><?php echo seconds_to_duration($liveMetrics['stream_duration']); ?></div>
                            <div class="metric-label">
                                <i class="fas fa-broadcast-tower metric-label-icon"></i>
                                Stream Duration
                            </div>
                        </div>
                        <?php endif; ?>
                    </div>
                    <?php if ($liveMetrics): ?>
                    <div style="margin-top: var(--spacing-lg); padding-top: var(--spacing-lg); border-top: 1px solid var(--border-secondary);">
                        <p style="font-size: 0.8125rem; color: var(--text-tertiary); text-align: center;">
                            <i class="fas fa-info-circle"></i> 
                            Tracking live metrics with <?php echo $liveMetrics['data_points']; ?> data point<?php echo $liveMetrics['data_points'] !== 1 ? 's' : ''; ?>
                            • Auto-updates every 30 seconds
                        </p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>

            <!-- Trends Tab -->
            <div class="tab-content" id="trends-tab">
                <div class="content-grid two-col">
                    <?php if (count($dailyStats) > 0): ?>
                    <div class="card boot-anim">
                        <div class="section-header">
                            <h2 class="section-title">30-Day Activity Trend</h2>
                        </div>
                        <div class="chart-container">
                            <canvas id="dailyTrendChart"></canvas>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($popularGames)): ?>
                    <div class="card boot-anim">
                        <div class="section-header">
                            <h2 class="section-title">Popular Games</h2>
                            <span style="font-size: 0.875rem; color: var(--text-tertiary);">Last 30 Days</span>
                        </div>
                        <div class="chart-container" style="height: 250px;">
                            <canvas id="gamesChart"></canvas>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (count($hourlyPatterns) > 0 && array_sum(array_column($hourlyPatterns, 'count')) > 0): ?>
                    <div class="card boot-anim">
                        <div class="section-header">
                            <h2 class="section-title">Streaming Schedule</h2>
                        </div>
                        <div class="chart-container" style="height: 250px;">
                            <canvas id="hourlyChart"></canvas>
                        </div>
                    </div>
                    <?php endif; ?>

                    <?php if (!empty($streamSchedule)): ?>
                    <div class="card boot-anim">
                        <div class="section-header">
                            <h2 class="section-title">Weekly Distribution</h2>
                        </div>
                        <div class="chart-container" style="height: 250px;">
                            <canvas id="scheduleChart"></canvas>
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Content Tab -->
            <div class="tab-content" id="content-tab">
                <div class="card boot-anim">
                    <div class="section-header">
                        <h2 class="section-title">Today's Broadcasts</h2>
                        <span style="font-size: 0.875rem; color: var(--text-tertiary);"><?php echo count($todayVods); ?> stream<?php echo count($todayVods) !== 1 ? 's' : ''; ?></span>
                    </div>
                    <?php if (!empty($todayVods)): ?>
                    <ul class="vod-list">
                        <?php foreach ($todayVods as $vod): ?>
                            <li class="vod-item">
                                <span class="vod-title"><?php echo htmlspecialchars($vod['title']); ?></span>
                                <div class="vod-meta">
                                    <span class="vod-meta-item">
                                        <i class="fas fa-eye" style="color: var(--primary);"></i>
                                        <?php echo isset($vod['view_count']) ? number_format($vod['view_count']) : 'N/A'; ?>
                                    </span>
                                    <span class="vod-meta-item">
                                        <i class="fas fa-clock" style="color: var(--text-tertiary);"></i>
                                        <?php 
                                            $vodCreatedAt = new DateTime($vod['created_at']);
                                            $vodCreatedAt->setTimezone($timezone);
                                            echo $vodCreatedAt->format('h:i A'); 
                                        ?> 
                                    </span>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                    <?php else: ?>
                    <div style="text-align: center; padding: 3rem 0; color: var(--text-tertiary);">
                        <i class="fas fa-video" style="font-size: 3rem; margin-bottom: 1rem; opacity: 0.3;"></i>
                        <p>No broadcasts found for today</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
        
    <?php include __DIR__ . '/includes/footer.php'; ?>
