<?php
// Twitch API Functions

/**
 * Format duration string to readable format
 */
function format_duration($duration) {
    $periods = array();
    preg_match('/(\d+)h/', $duration, $h);
    if (!empty($h)) {
        $hours = (int)$h[1];
        $periods[] = $hours . " hour" . ($hours > 1 ? "s" : "");
    }
    preg_match('/(\d+)m/', $duration, $m);
    if (!empty($m)) {
        $minutes = (int)$m[1];
        $periods[] = $minutes . " minute" . ($minutes > 1 ? "s" : "");
    }
    preg_match('/(\d+)s/', $duration, $s);
    if (!empty($s)) {
        $seconds = (int)$s[1];
        $periods[] = $seconds . " second" . ($seconds > 1 ? "s" : "");
    }
    return implode(", ", $periods) ?: "0 seconds";
}

/**
 * Convert duration string to seconds
 */
function duration_to_seconds($duration) {
    $seconds = 0;
    preg_match('/(\d+)h/', $duration, $h);
    if (!empty($h)) $seconds += (int)$h[1] * 3600;
    preg_match('/(\d+)m/', $duration, $m);
    if (!empty($m)) $seconds += (int)$m[1] * 60;
    preg_match('/(\d+)s/', $duration, $s);
    if (!empty($s)) $seconds += (int)$s[1];
    return $seconds;
}

/**
 * Format seconds to readable duration
 */
function seconds_to_duration($seconds) {
    $hours = floor($seconds / 3600);
    $minutes = floor(($seconds % 3600) / 60);
    return ($hours > 0 ? $hours . 'h ' : '') . ($minutes > 0 ? $minutes . 'm' : '0m');
}

/**
 * Make API request with error handling
 */
function api_request($url, $accessToken, $clientId) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array(
        "Client-ID: " . $clientId,
        "Authorization: Bearer " . $accessToken
    ));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        error_log("cURL Error: " . $error);
        return ['error' => 'Network error: ' . $error];
    }

    if ($httpCode == 429) {
        error_log("Rate limit exceeded for: " . $url);
        return ['error' => 'rate_limit', 'message' => 'Rate limit exceeded. Please try again later.'];
    }

    if ($httpCode != 200) {
        error_log("API Error ($httpCode): " . $response);
        return ['error' => 'api_error', 'code' => $httpCode, 'message' => 'API request failed'];
    }

    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log("JSON Error: " . json_last_error_msg());
        return ['error' => 'json_error', 'message' => 'Failed to parse API response'];
    }

    return $data;
}

/**
 * Get app access token
 */
function get_app_access_token($clientId, $clientSecret) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://id.twitch.tv/oauth2/token');
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query([
        'client_id' => $clientId,
        'client_secret' => $clientSecret,
        'grant_type' => 'client_credentials'
    ]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 10);
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode != 200) {
        error_log("Token Error ($httpCode): " . $response);
        return null;
    }
    
    $data = json_decode($response, true);
    return isset($data['access_token']) ? $data['access_token'] : null;
}

/**
 * Get user information
 */
function get_user_info($broadcasterLogin, $accessToken, $clientId) {
    $data = api_request("https://api.twitch.tv/helix/users?login=" . urlencode($broadcasterLogin), $accessToken, $clientId);
    if (isset($data['error'])) {
        return null;
    }
    return isset($data['data'][0]) ? $data['data'][0] : null;
}

/**
 * Get channel followers count
 */
function get_channel_followers($userId, $accessToken, $clientId) {
    $data = api_request("https://api.twitch.tv/helix/channels/followers?broadcaster_id=" . urlencode($userId), $accessToken, $clientId);
    if (isset($data['error'])) {
        return 0;
    }
    return isset($data['total']) ? $data['total'] : 0;
}

/**
 * Get current stream information
 */
function get_stream_info($userId, $accessToken, $clientId) {
    $data = api_request("https://api.twitch.tv/helix/streams?user_id=" . urlencode($userId), $accessToken, $clientId);
    if (isset($data['error'])) {
        return null;
    }
    return isset($data['data'][0]) ? $data['data'][0] : null;
}

/**
 * Track live stream metrics (viewer counts over time)
 */
function track_live_stream_metrics($userId, $viewerCount, $cacheDir) {
    if (!file_exists($cacheDir)) {
        @mkdir($cacheDir, 0755, true);
    }
    
    $trackingFile = $cacheDir . '/live_metrics_' . $userId . '.json';
    $currentTime = time();
    
    // Load existing tracking data
    $trackingData = ['viewer_counts' => [], 'started_at' => null, 'last_update' => null];
    if (file_exists($trackingFile)) {
        $existing = json_decode(file_get_contents($trackingFile), true);
        if ($existing) {
            $trackingData = $existing;
        }
    }
    
    // If stream just started or no data exists, initialize
    if (empty($trackingData['viewer_counts']) || $trackingData['last_update'] < ($currentTime - 300)) {
        // If last update was more than 5 minutes ago, assume new stream
        $trackingData = [
            'viewer_counts' => [],
            'started_at' => $currentTime,
            'last_update' => $currentTime
        ];
    }
    
    // Add current viewer count with timestamp
    $trackingData['viewer_counts'][] = [
        'timestamp' => $currentTime,
        'viewers' => (int)$viewerCount
    ];
    
    // Keep only last 24 hours of data (1440 minutes, assuming 1 data point per minute)
    $cutoffTime = $currentTime - (24 * 60 * 60);
    $trackingData['viewer_counts'] = array_filter($trackingData['viewer_counts'], function($entry) use ($cutoffTime) {
        return $entry['timestamp'] >= $cutoffTime;
    });
    $trackingData['viewer_counts'] = array_values($trackingData['viewer_counts']); // Re-index
    
    $trackingData['last_update'] = $currentTime;
    
    // Save tracking data
    file_put_contents($trackingFile, json_encode($trackingData));
    
    return $trackingData;
}

/**
 * Get live stream metrics (average, peak, etc.)
 */
function get_live_stream_metrics($userId, $cacheDir) {
    $trackingFile = $cacheDir . '/live_metrics_' . $userId . '.json';
    
    if (!file_exists($trackingFile)) {
        return null;
    }
    
    $trackingData = json_decode(file_get_contents($trackingFile), true);
    if (!$trackingData || empty($trackingData['viewer_counts'])) {
        return null;
    }
    
    $viewerCounts = array_column($trackingData['viewer_counts'], 'viewers');
    
    if (empty($viewerCounts)) {
        return null;
    }
    
    $metrics = [
        'average_viewers' => (int)round(array_sum($viewerCounts) / count($viewerCounts)),
        'peak_viewers' => max($viewerCounts),
        'current_viewers' => end($viewerCounts),
        'min_viewers' => min($viewerCounts),
        'data_points' => count($viewerCounts),
        'started_at' => $trackingData['started_at'] ?? null,
        'stream_duration' => isset($trackingData['started_at']) ? (time() - $trackingData['started_at']) : 0
    ];
    
    return $metrics;
}

/**
 * Clear live stream metrics (when stream ends)
 */
function clear_live_stream_metrics($userId, $cacheDir) {
    $trackingFile = $cacheDir . '/live_metrics_' . $userId . '.json';
    if (file_exists($trackingFile)) {
        @unlink($trackingFile);
    }
}

/**
 * Get VODs for a user
 */
function get_vods($userId, $accessToken, $clientId, $first = 10, $type = 'archive') {
    $data = api_request("https://api.twitch.tv/helix/videos?user_id=" . urlencode($userId) . "&first=" . $first . "&type=" . $type, $accessToken, $clientId);
    if (isset($data['error'])) {
        return [];
    }
    return isset($data['data']) ? $data['data'] : [];
}

/**
 * Get all VODs (with pagination)
 */
function get_all_vods($userId, $accessToken, $clientId, $maxResults = 100, $type = 'archive') {
    $allVods = [];
    $cursor = null;
    $first = min(100, $maxResults);
    
    do {
        $url = "https://api.twitch.tv/helix/videos?user_id=" . urlencode($userId) . "&first=" . $first . "&type=" . $type;
        if ($cursor) {
            $url .= "&after=" . urlencode($cursor);
        }
        
        $data = api_request($url, $accessToken, $clientId);
        if (isset($data['error'])) {
            break;
        }
        
        if (isset($data['data'])) {
            $allVods = array_merge($allVods, $data['data']);
        }
        
        $cursor = isset($data['pagination']['cursor']) ? $data['pagination']['cursor'] : null;
        
        if (count($allVods) >= $maxResults) {
            break;
        }
    } while ($cursor);
    
    return array_slice($allVods, 0, $maxResults);
}

/**
 * Calculate analytics from VODs
 */
function calculate_vod_analytics($vods, $timezone) {
    $analytics = [
        'total_vods' => count($vods),
        'total_duration' => 0,
        'total_views' => 0,
        'average_views' => 0,
        'peak_views' => 0,
        'games' => []
    ];
    
    foreach ($vods as $vod) {
        // Duration - handle missing duration gracefully
        if (isset($vod['duration'])) {
            $duration = duration_to_seconds($vod['duration']);
            $analytics['total_duration'] += $duration;
        }
        
        // Views
        $views = isset($vod['view_count']) ? (int)$vod['view_count'] : 0;
        $analytics['total_views'] += $views;
        if ($views > $analytics['peak_views']) {
            $analytics['peak_views'] = $views;
        }
        
        // Games
        if (isset($vod['game_name']) && !empty($vod['game_name'])) {
            $gameName = $vod['game_name'];
            if (!isset($analytics['games'][$gameName])) {
                $analytics['games'][$gameName] = 0;
            }
            $analytics['games'][$gameName]++;
        }
    }
    
    // Calculate average views - only if we have VODs
    if ($analytics['total_vods'] > 0) {
        $analytics['average_views'] = (int)round($analytics['total_views'] / $analytics['total_vods']);
    }
    
    // Sort games by count
    arsort($analytics['games']);
    
    return $analytics;
}

/**
 * Filter VODs by date range
 */
function filter_vods_by_date($vods, $startDate, $endDate, $timezone) {
    $filtered = [];
    $start = new DateTime($startDate, $timezone);
    $start->setTime(0, 0, 0); // Start of day
    $end = new DateTime($endDate, $timezone);
    $end->setTime(23, 59, 59); // End of day
    
    foreach ($vods as $vod) {
        if (!isset($vod['created_at'])) {
            continue;
        }
        $vodDate = new DateTime($vod['created_at']);
        $vodDate->setTimezone($timezone);
        
        if ($vodDate >= $start && $vodDate <= $end) {
            $filtered[] = $vod;
        }
    }
    
    return $filtered;
}

/**
 * Cache helper functions
 */
function get_cache_key($prefix, $params) {
    return $prefix . '_' . md5(serialize($params));
}

function get_cached_data($key, $ttl, $cacheDir) {
    if (!file_exists($cacheDir)) {
        @mkdir($cacheDir, 0755, true);
    }
    
    $cacheFile = $cacheDir . '/' . $key . '.cache';
    
    if (!file_exists($cacheFile)) {
        return null;
    }
    
    $cacheData = json_decode(file_get_contents($cacheFile), true);
    
    if (!$cacheData || !isset($cacheData['expires']) || !isset($cacheData['data'])) {
        return null;
    }
    
    if (time() > $cacheData['expires']) {
        @unlink($cacheFile);
        return null;
    }
    
    return $cacheData['data'];
}

function set_cached_data($key, $data, $ttl, $cacheDir) {
    if (!file_exists($cacheDir)) {
        @mkdir($cacheDir, 0755, true);
    }
    
    $cacheFile = $cacheDir . '/' . $key . '.cache';
    $cacheData = [
        'expires' => time() + $ttl,
        'data' => $data
    ];
    
    file_put_contents($cacheFile, json_encode($cacheData));
}

/**
 * Calculate daily statistics for time series charts
 */
function calculate_daily_stats($vods, $timezone, $days = 30) {
    $dailyStats = [];
    $endDate = new DateTime('now', $timezone);
    $endDate->setTime(23, 59, 59);
    $startDate = clone $endDate;
    $startDate->modify("-$days days");
    $startDate->setTime(0, 0, 0);
    
    // Initialize all days
    $current = clone $startDate;
    while ($current <= $endDate) {
        $dateKey = $current->format('Y-m-d');
        $dailyStats[$dateKey] = [
            'date' => $dateKey,
            'streams' => 0,
            'total_views' => 0,
            'total_duration' => 0,
            'peak_viewers' => 0
        ];
        $current->modify('+1 day');
    }
    
    // Fill in actual data
    foreach ($vods as $vod) {
        if (!isset($vod['created_at'])) {
            continue;
        }
        $vodDate = new DateTime($vod['created_at']);
        $vodDate->setTimezone($timezone);
        $dateKey = $vodDate->format('Y-m-d');
        
        if (isset($dailyStats[$dateKey])) {
            $dailyStats[$dateKey]['streams']++;
            $views = isset($vod['view_count']) ? (int)$vod['view_count'] : 0;
            $dailyStats[$dateKey]['total_views'] += $views;
            $duration = isset($vod['duration']) ? duration_to_seconds($vod['duration']) : 0;
            $dailyStats[$dateKey]['total_duration'] += $duration;
            if ($views > $dailyStats[$dateKey]['peak_viewers']) {
                $dailyStats[$dateKey]['peak_viewers'] = $views;
            }
        }
    }
    
    return array_values($dailyStats);
}

/**
 * Calculate hourly streaming patterns
 */
function calculate_hourly_patterns($vods, $timezone) {
    $hourlyStats = array_fill(0, 24, ['count' => 0, 'total_views' => 0]);
    
    foreach ($vods as $vod) {
        if (!isset($vod['created_at'])) {
            continue;
        }
        $vodDate = new DateTime($vod['created_at']);
        $vodDate->setTimezone($timezone);
        $hour = (int)$vodDate->format('H');
        
        if ($hour >= 0 && $hour < 24) {
            $hourlyStats[$hour]['count']++;
            $views = isset($vod['view_count']) ? (int)$vod['view_count'] : 0;
            $hourlyStats[$hour]['total_views'] += $views;
        }
    }
    
    return $hourlyStats;
}

/**
 * Calculate growth metrics
 */
function calculate_growth_metrics($vods, $timezone) {
    $now = new DateTime('now', $timezone);
    $now->setTime(23, 59, 59);
    $weekAgo = clone $now;
    $weekAgo->modify('-7 days');
    $weekAgo->setTime(0, 0, 0);
    $twoWeeksAgo = clone $weekAgo;
    $twoWeeksAgo->modify('-7 days');
    $weekAgoEnd = clone $weekAgo;
    $weekAgoEnd->modify('-1 second');
    
    $thisWeek = filter_vods_by_date($vods, $weekAgo->format('Y-m-d'), $now->format('Y-m-d'), $timezone);
    $lastWeek = filter_vods_by_date($vods, $twoWeeksAgo->format('Y-m-d'), $weekAgoEnd->format('Y-m-d'), $timezone);
    
    $thisWeekViews = 0;
    foreach ($thisWeek as $vod) {
        $thisWeekViews += isset($vod['view_count']) ? (int)$vod['view_count'] : 0;
    }
    
    $lastWeekViews = 0;
    foreach ($lastWeek as $vod) {
        $lastWeekViews += isset($vod['view_count']) ? (int)$vod['view_count'] : 0;
    }
    
    $viewsGrowth = 0;
    if ($lastWeekViews > 0) {
        $viewsGrowth = (($thisWeekViews - $lastWeekViews) / $lastWeekViews) * 100;
    } elseif ($thisWeekViews > 0 && $lastWeekViews == 0) {
        $viewsGrowth = 100; // New streams (infinite growth from 0)
    }
    
    return [
        'this_week_views' => $thisWeekViews,
        'last_week_views' => $lastWeekViews,
        'views_growth_percent' => round($viewsGrowth, 1),
        'this_week_streams' => count($thisWeek),
        'last_week_streams' => count($lastWeek)
    ];
}

/**
 * Get stream schedule patterns
 */
function get_stream_schedule($vods, $timezone) {
    $schedule = [];
    
    foreach ($vods as $vod) {
        if (!isset($vod['created_at'])) {
            continue;
        }
        $vodDate = new DateTime($vod['created_at']);
        $vodDate->setTimezone($timezone);
        $dayOfWeek = $vodDate->format('l'); // Monday, Tuesday, etc.
        
        if (!isset($schedule[$dayOfWeek])) {
            $schedule[$dayOfWeek] = 0;
        }
        $schedule[$dayOfWeek]++;
    }
    
    // Order by day of week
    $daysOrder = ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday', 'Sunday'];
    $orderedSchedule = [];
    foreach ($daysOrder as $day) {
        if (isset($schedule[$day])) {
            $orderedSchedule[$day] = $schedule[$day];
        }
    }
    
    return $orderedSchedule;
}

